import os
import re
from sets import Set as set

LOG_PATH = '../log'
RESULT_PATH = '/analysis/concurrent'

FUTURE_START_TYPE = 'Created future'
FUTURE_FINISH_TYPE = 'Future is done'
TASK_START_TYPE = 'Started task'
TASK_FINISH_TYPE = 'Finished task'

taskEvolutionSteps = [FUTURE_START_TYPE, TASK_START_TYPE, TASK_FINISH_TYPE, FUTURE_FINISH_TYPE]

errors = []


#---------------------------------------------------------------------------------------------------------
# This method will create the given dir if it does not already exist.

def _mkdir(aNewdir):
    """works the way a good mkdir should :)
        - already exists, silently complete
        - regular file in the way, raise an exception
        - parent directory(ies) does not exist, make them as well"""
    
    if os.path.isdir(aNewdir):
        pass
    elif os.path.isfile(aNewdir):
        raise OSError("a file with the same name as the desired " \
                      "dir, '%s', already exists." % aNewdir)
    else:
        head, tail = os.path.split(aNewdir)
        if head and not os.path.isdir(head):
            _mkdir(head)
        if tail:
            os.mkdir(aNewdir)

#---------------------------------------------------------------------------------------------------------

def findFiles(aPath, aPrefix, aExtension):
    """This method will find all the files in the given path, which contain the given 
    prefix and end with the given extension."""

    allFiles = os.listdir(aPath)
    retVal = []
    for file in allFiles:
        if (file.find(aPrefix) != -1) & (file.find(aExtension) != -1):
            retVal += [file]
            
    return retVal


#---------------------------------------------------------------------------------------------------------
class FutureTaskMsg:
    """Represents a SNMP message.
    Contains a message type and its boddy as list of lines.
    """
    
    #Constructor
    def __init__(self, aLine, aFile):
        #The type of the aLine
        self.myType = ''
        #The id of the future/task
        self.myId = 0
        #The line
        self.myLine = aLine
        #The log file
        self.myFile = aFile

        if (aLine.find(FUTURE_START_TYPE) > -1):
            self.myType = FUTURE_START_TYPE
        elif (aLine.find(FUTURE_FINISH_TYPE) > -1):
            self.myType = FUTURE_FINISH_TYPE
        elif (aLine.find(TASK_START_TYPE) > -1):
            self.myType = TASK_START_TYPE
        elif (aLine.find(TASK_FINISH_TYPE) > -1):
            self.myType = TASK_FINISH_TYPE
            
        self.initID(aLine)
        
        
    def initID(self, line):
        """Extract the ID"""
        regexObject = re.compile('.+' + self.myType + '\s+\(' + '(\d+)' + '\).*')
        
        #if (regexObject.match(line)):
        #    if (self.myType == ''):
        #        print 'ERROR: match was found with no msg type ' + line
        
        if (regexObject.match(line)):
            matchObject = regexObject.search(line)
            self.myId = matchObject.group(1)
            
                
#---------------------------------------------------------------------------------------------------------

def getByTypeAndID(aFutureTaskMsgs, aId, aType):
    """This method return a msg with the specified id and type.
    @param id: a msg ID
    @param aType: a msg type (FUTURE_START_TYPE or TASK_START_TYPE etc.).
    @return: a msg with the same type and ID or None if not found.
    @rtype: FutureTaskMsg"""
    filteredMsgs = []
    for msg in aFutureTaskMsgs:
        if aId == msg.myId:
            if aType.find(msg.myType) == 0:
                return msg

    return None
#---------------------------------------------------------------------------------------------------------

def filterByType(aFutureTaskMsgs, aType):
    """This method return a msg with the specified id and type.
    @param aType: a msg type (FUTURE_START_TYPE or TASK_START_TYPE etc.).
    @return: a msg with the same type and ID or None if not found.
    @rtype: list of FutureTaskMsg"""
    filteredMsgs = []
    for msg in aFutureTaskMsgs:
        if aType.find(msg.myType) == 0:
            filteredMsgs.append(msg)

    return filteredMsgs
#---------------------------------------------------------------------------------------------------------

# This method will print the exceptions in the array to the given file.
def saveAllToFiles(aSnmpMsgList):
    
    for ipDest in destIPSet:
        filteredMsgs = filterByDestination(ipDest, aSnmpMsgList)
    
    
        # Create the dir.
        path = LOG_PATH + RESULT_PATH
        _mkdir(path)
        
        # Create the new file and save the snmp messages in it.
        newFileName = path + '/snmp_messages_' + ipDest + '.txt'
        if (os.path.isfile(newFileName)):
            os.remove(newFileName)
        
        print 'writing to ' + newFileName
        f = open(newFileName, 'w')
        
        for snmpMsg in filteredMsgs:
            f.write(snmpMsg.getSnmpMessage())
            f.write('\n')
            f.write('*' * 120)
            f.write('\n')
    
        f.flush()
        f.close()
        
        # Create the summary file
        saveSummaryToFile(ipDest, filteredMsgs)

#---------------------------------------------------------------------------------------------------------

def saveSummaryToFile(destIP, aSnmpMsgList):
    # Create the dir.
    path = LOG_PATH + RESULT_PATH
    _mkdir(path)
    
    # Create the new file and save the snmp messages in it.
    newFileName = path + '/summary_' + destIP + '.txt'
    if (os.path.isfile(newFileName)):
        os.remove(newFileName)
    print 'writing to ' + newFileName
    f = open(newFileName, 'w')
    
    setCount = 0
    getCount = 0
    getNextCount = 0
    getBulkCount = 0
    unknownCount = 0
    
    for snmpMsg in aSnmpMsgList:
        if (snmpMsg.type == SNMP_SET_MSG_TYPE):
            setCount += 1
        elif (snmpMsg.type == SNMP_GET_MSG_TYPE):
            getCount += 1
        elif (snmpMsg.type == SNMP_GETNEXT_MSG_TYPE):
            getNextCount += 1
        elif (snmpMsg.type == SNMP_GETBULK_MSG_TYPE):
            getBulkCount += 1
        elif (snmpMsg.type == SNMP_UNKNOWN_MSG_TYPE):
            unknownCount += 1
        else:
            raise ValueError("Illegal state: SnmpMsg object with no type")
            
            
    f.write('Summary of timeout SNMP messages:\n')
    f.write('---------------------------------\n\n')
    f.write('Number of set commands: ' + str(setCount) + '\n\n')
    f.write('Number of get commands: ' + str(getCount) + '\n\n')
    f.write('Number of get-next commands: ' + str(getNextCount) + '\n\n')
    f.write('Number of get-bulk commands: ' + str(getBulkCount) + '\n\n')
    f.write('Number of unknown type commands: ' + str(unknownCount) + '\n\n')
    
    f.write('\n\n\nError summary:\n')
    f.write('---------------------------------\n')
    for line in errors:
        f.write(line + '\n')

    f.flush()
    f.close()

#---------------------------------------------------------------------------------------------------------
def saveUnfinishedFuturesReport(aMsgList):
    """Save a report of the unfinished futures to a file
    @param aMsgList: a list of FutureTaskMsg"""
    unfinishedFutureMsgs = []
    
    createdFutureMsgs = filterByType(aMsgList, FUTURE_START_TYPE)
    finishedFutureMsgs = filterByType(aMsgList, FUTURE_FINISH_TYPE)
    
    if len(createdFutureMsgs) == 0:
        print 'There are no created Futures'
        return
    
    print 'Searching for unfinished futures...'
    for createMsg in createdFutureMsgs:
        if getByTypeAndID(finishedFutureMsgs, createMsg.myId, FUTURE_FINISH_TYPE) == None:
            unfinishedFutureMsgs.append(createMsg)
            
    # Create the dir.
    path = LOG_PATH + RESULT_PATH
    _mkdir(path)
    
    # Create the new file and save the snmp messages in it.
    newFileName = path + '/unfinished_futures.txt'
    if os.path.isfile(newFileName):
        os.remove(newFileName)
    print 'Writing to ' + newFileName
    f = open(newFileName, 'w')
    
    f.write('Summary of unfinished futures:\n')
    
    for unfinishedFuture in unfinishedFutureMsgs:
        f.write('Future ID ')
        f.write(unfinishedFuture.myId)
        f.write(' found unfinished in file ')
        f.write(unfinishedFuture.myFile)
        f.write(' , record: ')
        f.write(unfinishedFuture.myLine)

    f.flush()
    f.close()

#---------------------------------------------------------------------------------------------------------

# This method returns a string report of all snmp retries from the given file.

def getAllFutureTaskFromFile(aFile):
    """This method analyze the specified file from bottom to top and return
        a list of SnmpMsg objects which encountered timeout.
        @param aFile: a file full path.
        @return: a list of SnmpMsg objects.
    """
    # get file lines and reverse them
    lines = open(aFile).readlines()
    print('Processing ' + aFile)
    # A list of SnmpMsg objects which encountered timeout
    msgList = []
    # loop on file lines
    for line in lines:
        msgList.append(FutureTaskMsg(line, aFile))
        
    return msgList

#---------------------------------------------------------------------------------------------------------

#MAIN
print 'FIND UNFINISHED FUTURES'

# Get the snmp logs
prefix = 'server'
extension = '.log'
logFiles = findFiles(LOG_PATH, prefix, extension)
print(logFiles)

#A list of SnmpMsg objects which is the result of the files analyzing
msgList = []
#Loop on files and analyze them
for file in logFiles:
    msgList.extend(getAllFutureTaskFromFile(LOG_PATH + '/' + file))

#print unfinished futures report
saveUnfinishedFuturesReport(msgList)
    
print('end')
