import os
import re
import zlib
import zipfile
from sets import Set as set

LOG_PATH = '../log'
RESULT_PATH = '/analysis/timeout'
ANALYSIS_DIR_NAME = '/analysis'
DETAILS_DIR_NAME = '/details'
#SNMP PDU properties
SNMP_VERSION = 'version'
SNMP_COMMUNITY = 'community'
PDU_TYPE = 'pduType'
REQUEST_ID = 'requestId'
ERROR_STATUS = 'errorStatus'
ERROR_INDEX = 'errorIndex'
VAR_BIND = 'varBindList'
SNMP_MSG_END = '}'

#any SNMP message starts with -
SNMP_MSG_HEADER = '|DEBUG|SENDING SNMP MSG:'
SNMP_SET_MSG_TYPE = 'setRequest'
SNMP_GET_MSG_TYPE = 'getRequest'
SNMP_GETNEXT_MSG_TYPE = 'getNextRequest'
SNMP_GETBULK_MSG_TYPE = 'getBulkRequest'
SNMP_UNKNOWN_MSG_TYPE = 'UNKNOWN TYPE'
KEEP_ALIVE_MSG_TYPE = 'KEEP_ALIVE_MSG_TYPE'

#timeout idetifier - when timeout occures snmp log contains this string
#right after the snmp PDU
TIMEOUT_IDENTIFIER = '|DEBUG|Firing TimeoutItem: ['

#To hold all errors which encountered in the process
errors = []

#This is a set of the IP destinations that encountered a timeout
destIPSet = set([])

###################
#### Regex tools ##
###################
#Regex for 
REGEX_DestinationIP = re.compile('.*address=(.*):.*')
REGEX_MsgRequestID = re.compile('.*' + REQUEST_ID + '=(.*)')
REGEX_TimeoutRequestID = re.compile('.*reqid=(\d*).*')


#---------------------------------------------------------------------------------------------------------
# This method will create the given dir if it does not already exist.

def _mkdir(aNewdir):
    #works the way a good mkdir should :)
    #    - already exists, silently complete
    #    - regular file in the way, raise an exception
    #    - parent directory(ies) does not exist, make them as well
    
    if os.path.isdir(aNewdir):
        pass
    elif os.path.isfile(aNewdir):
        raise OSError("a file with the same name as the desired " \
                      "dir, '%s', already exists." % aNewdir)
    else:
        head, tail = os.path.split(aNewdir)
        if head and not os.path.isdir(head):
            _mkdir(head)
        if tail:
            os.mkdir(aNewdir)


#---------------------------------------------------------------------------------------------------------
class SnmpMsg:
    """Represents a SNMP message.
    Contains a message type and its boddy as list of lines.
    """
    
    #Constructor
    def __init__(self, aSnmpMsgLines, aFile):
        #The type of the message
        self.type = SNMP_UNKNOWN_MSG_TYPE
        #The boddy of the message
        self.boddy = []
        #The IP address of the SNMP agent
        self.destIP = ''
        #The SNMP request ID
        self.requestId = 0
        
        #Init the attributes
        for line in aSnmpMsgLines:
            #Build the msg boddy
            self.boddy.append(line)
            #Set the msg type
            if (line.find(PDU_TYPE) > -1):
                if (line.find(SNMP_SET_MSG_TYPE) > -1):
                    self.type = SNMP_SET_MSG_TYPE
                elif (line.find(SNMP_GET_MSG_TYPE) > -1):
                    self.type = SNMP_GET_MSG_TYPE
                elif (line.find(SNMP_GETNEXT_MSG_TYPE) > -1):
                    self.type = SNMP_GETNEXT_MSG_TYPE
                elif (line.find(SNMP_GETBULK_MSG_TYPE) > -1):
                    self.type = SNMP_GETBULK_MSG_TYPE
                else:
                    self.type = SNMP_UNKNOWN_MSG_TYPE
                    errors.append('Encountered unkown type ' + line + ' in file ' + aFile)
                    
            if (line.find(REQUEST_ID)):
                self.initRequestID(line)
            
            if (line.find(SNMP_MSG_END) == 0):
                break
            
        #This is the last line, it contains the TIMEOUT_IDENTIFIER and the IP of the message
        #destination
        self.boddy.append(aSnmpMsgLines[len(aSnmpMsgLines) - 1])
        self.initDestinationIP(aSnmpMsgLines[len(aSnmpMsgLines) - 1])

        
    def initDestinationIP(self, line):
        """Extract the destination IP address from the specified line parameter.
        The specified line is the TIMEOUT_IDENTIFIER line"""
        if (REGEX_DestinationIP.match(line)):
            matchObject = REGEX_DestinationIP.search(line)
            self.destIP = matchObject.group(1)
            
    def initRequestID(self, line):
        """Extract the request ID from the REQUEST_ID line."""
        if (REGEX_MsgRequestID.match(line)):
            matchObject = REGEX_MsgRequestID.search(line)
            self.requestId = matchObject.group(1)
        
            
    def getSnmpMessage(self):
        """Get the message line in one String"""
        allMsgString = ''
        for line in self.boddy:
            allMsgString += line
        
        return allMsgString
                
#---------------------------------------------------------------------------------------------------------

#This method helps to filter a SnmpMsg by its destination IP.
def filterByDestination(dest, snmpMsgs):
    filteredMsgs = []
    for msg in snmpMsgs:
        if (dest == msg.destIP):
            filteredMsgs.append(msg)

    return filteredMsgs
#---------------------------------------------------------------------------------------------------------

# This method will print the exceptions in the array to the given file.
def saveAllToFiles(aSnmpMsgList,path):
    # Create the dir.
    _mkdir(path)
        
    for ipDest in destIPSet:
        filteredMsgs = filterByDestination(ipDest, aSnmpMsgList)
    
    
        # Create the new file and save the snmp messages in it.
        newFileName = path + '/snmp_messages_' + ipDest + '.txt'
        if (os.path.isfile(newFileName)):
            os.remove(newFileName)
        
        print 'writing to ' + newFileName
        f = open(newFileName, 'w')
        
        for snmpMsg in filteredMsgs:
            f.write(snmpMsg.getSnmpMessage())
            f.write('\n')
            f.write('*' * 120)
            f.write('\n')
    
        f.flush()
        f.close()
        
        # Create the summary file
        saveSummaryToFile(ipDest, filteredMsgs,path)
    destIPSet.clear()

#---------------------------------------------------------------------------------------------------------

def saveSummaryToFile(destIP, aSnmpMsgList,path):
    
    # Create the new file and save the snmp messages in it.
    newFileName = path + '/summary_' + destIP + '.txt'
    if (os.path.isfile(newFileName)):
        os.remove(newFileName)
    print 'writing to ' + newFileName
    f = open(newFileName, 'w')
    
    setCount = 0
    getCount = 0
    getNextCount = 0
    getBulkCount = 0
    unknownCount = 0
    
    for snmpMsg in aSnmpMsgList:
        if (snmpMsg.type == SNMP_SET_MSG_TYPE):
            setCount += 1
        elif (snmpMsg.type == SNMP_GET_MSG_TYPE):
            getCount += 1
        elif (snmpMsg.type == SNMP_GETNEXT_MSG_TYPE):
            getNextCount += 1
        elif (snmpMsg.type == SNMP_GETBULK_MSG_TYPE):
            getBulkCount += 1
        elif (snmpMsg.type == SNMP_UNKNOWN_MSG_TYPE):
            unknownCount += 1
        else:
            raise ValueError("Illegal state: SnmpMsg object with no type")
            
            
    f.write('Summary of timeout SNMP messages:\n')
    f.write('---------------------------------\n\n')
    f.write('Number of set commands: ' + str(setCount) + '\n\n')
    f.write('Number of get commands: ' + str(getCount) + '\n\n')
    f.write('Number of get-next commands: ' + str(getNextCount) + '\n\n')
    f.write('Number of get-bulk commands: ' + str(getBulkCount) + '\n\n')
    f.write('Number of unknown type commands: ' + str(unknownCount) + '\n\n')
    
    f.write('\n\n\nError summary:\n')
    f.write('---------------------------------\n')
    for line in errors:
        f.write(line + '\n')
        errors.remove(line)
    f.flush()
    f.close()

#---------------------------------------------------------------------------------------------------------

# This method returns a string report of all snmp retries from the given file.

def getSnmpRetriesReportsFromFile(aFile):
    """This method analyze the specified file from bottom to top and return
        a list of SnmpMsg objects which encountered timeout.
        @param aFile: a file full path.
        @return: a list of SnmpMsg objects.
    """
    # get file lines and reverse them
    lines = open(aFile).readlines()
    lines.reverse()
    print('Processing ' + aFile)
    # A list of SnmpMsg objects which encountered timeout
    timeoutSnmpMsgList = []
    # temporary array of log lines which contain the timeout-pdu
    snmpMsgReveredLines = []
    
    # A state holder to analyse the file
    timeoutFound = False
    timeoutLine = ''
    timeoutReqId = 0
    timeoutDestIP = ''
    # loop on file revered lines
    for line in lines:
        # Take the line which contains the timeout identifier.
        if (line.find(TIMEOUT_IDENTIFIER) > -1):
            timeoutFound = True
            timeoutLine = line
            #extract the request ID from the timeout line
            if (REGEX_TimeoutRequestID.match(line)):
                matchObject = REGEX_TimeoutRequestID.search(line)
                timeoutReqId = matchObject.group(1)
            #extract the destination IP from the timeout line
            if (REGEX_DestinationIP.match(line)):
                matchObject = REGEX_DestinationIP.search(line)
                timeoutDestIP = matchObject.group(1)
                #add this destination IP to the global set
                destIPSet.add(timeoutDestIP)
            #print 'timeout found in file ' + aFile + ' IP=' + timeoutDestIP + ' ID=%s', timeoutReqId

        # Take the message header line
        if (line.find(SNMP_MSG_HEADER) > -1 & timeoutFound == True):
            #create a SnmpMsg from the lines to the SNMP_MSG_HEADER
            snmpMsgReveredLines.append(line)
            snmpMsgReveredLines.reverse()
            snmpMsgReveredLines.append(timeoutLine)
            snmpMsg = SnmpMsg(snmpMsgReveredLines, aFile)
            
            if (snmpMsg.requestId == timeoutReqId):
            	#print 'message found %s', timeoutReqId
                timeoutSnmpMsgList.append(snmpMsg)
                timeoutFound = False
                
            snmpMsgReveredLines = []
        # Take all line between the header and the timeout identifier.
        if (timeoutFound):
            snmpMsgReveredLines.append(line)
    if (timeoutFound == True):
        errors.append('Found a timeout in ' + aFile + ' but SNMP message not found')
    
    # Reverse the SnmpMsg objects so they will appear in the same order of their
    # appearance in the file
    timeoutSnmpMsgList.reverse()
    print 'number of timeout in file ' + aFile + ' is ', len(timeoutSnmpMsgList)
    return timeoutSnmpMsgList

#---------------------------------------------------------------------------------------------------------

# This method will find all the files in the given path, which contain the given 
# prefix and end with the given extension.

def findFiles(aPath, aPrefix, aExtension):
    allFiles = os.listdir(aPath)
    retVal = []
    for file in allFiles:
        if (file.find(aPrefix) != -1) & (file.find(aExtension) != -1):
            retVal += [file]
            
    return retVal

#---------------------------------------------------------------------------------------------------------

def removeFiles(path, prefix, extension):
    """   This method will remove all the files starting
    with the given prefix and ending with the given 
    extension in the given path.
     
    @param path: The path to remove the files from .
    @param prefix: The prefix of the file name to remove.
    @param extension: The extension of the file to remove.
    """   
    
    # Find all the files and remove them one by one.
    allFiles = findFiles(path, prefix, extension)
    
    for fileName in allFiles:
        fullpath = path + '/' + fileName
        
        if os.path.isfile(fullpath):
            f=os.remove
            rmgeneric(fullpath, f)
	    
#----------------------------------------------------------------------------------
def unzip(zip, dir):
    for name in zip.namelist():
	 fname= dir + '/' + name
         FI = open(fname,"w")
         FI.writelines(zip.read(name))
         FI.close()

        
#------------------------------------------------------------------------------
def rmgeneric(path, __func__):
    """   Clean up a directory tree from root. The directory need not be empty.
    The starting directory is not deleted.
    """

    try:
        __func__(path)
    except OSError, (errno, strerror):
        print ERROR_STR % {'path' : path, 'error': strerror }
#------------------------------------------------------------------------------
def analyzeUnzippedSnmpExpLogs(newDir):
	prefix = 'snmp'
	extension = '.log'
	snmpFiles = findFiles(newDir, prefix, extension)
	print(snmpFiles)
	snmpMsgList = []
	for file in snmpFiles:
            snmpMsgList.extend(getSnmpRetriesReportsFromFile(newDir + '/' + file))
	saveAllToFiles(snmpMsgList,newDir)
#------------------------------------------------------------------------------

#MAIN
print 'SNMP Retries analyzing...'
print 'THIS SCRIPT SHOULD BE COPIED TO THE "SERVER/SCRIPTS" DIRECTORY'
print 'THE LOG_PATH VARIABLE NEEDS TO CHANGE TO ../LOG NOT ./LOG'

# Get the snmp logs
prefix = 'snmp'
extension = '.log'
zipExtention = '.zip'
snmpFiles = findFiles(LOG_PATH, prefix, extension)
print(snmpFiles)

#A list of SnmpMsg objects which is the result of the files analyzing
snmpMsgList = []
#Loop on files and analyze them
for file in snmpFiles:
    snmpMsgList.extend(getSnmpRetriesReportsFromFile(LOG_PATH + '/' + file))
#print snmp messages and summary to files
saveAllToFiles(snmpMsgList,LOG_PATH + RESULT_PATH + '/current' )

snmpZipFiles= findFiles(LOG_PATH,prefix,zipExtention)
for fileName in snmpZipFiles:
	 if (zipfile.is_zipfile(LOG_PATH + '/' + fileName) == False) :
		 continue


	 newDir = LOG_PATH + ANALYSIS_DIR_NAME + '/' +'timeout' + '/'+ fileName
	 newDir = newDir[0:len(newDir) - 4]  #remove '.zip' from dir name
	 print 'Analyzing ' + fileName
	 _mkdir(newDir)
	 
	 # Unzip the file
	 curZip = zipfile.ZipFile(LOG_PATH + '/' + fileName, 'r')
	 unzip(curZip, newDir)
	 curZip.close

	 # Analyze the snmp logs.
	 analyzeUnzippedSnmpExpLogs(newDir)

	 # Remove the unnecessary files.
	 removeFiles(newDir, 'snmp', 'log')

print('end')
